;;; -*- Mode:Common-Lisp; Package:TV; Fonts:(CPTFONT HL10B HL12BI HL12B CPTFONTB); Base:81; *Patch-File:T -*-

;1;;                     RESTRICTED RIGHTS LEGEND          *
;1;; Use, duplication, or disclosure by the Government is subject to*
;1;; restrictions as set forth in subdivision (b)(3)(ii) of the Rights in*
;1;; Technical Data and Computer Software clause at 52.227-7013.*
;1;;                   TEXAS INSTRUMENTS INCORPORATED.*
;1;;                            P.O. BOX 149149*
;1;;                         AUSTIN, TEXAS 78714-9149*
;1;;                             MS 2151*
;1;; Copyright (C) 1984-1989 Texas Instruments Incorporated. All rights reserved.*

;; PEEK -- displays status information about the Lisp Machine
;; 03-14-89 DAB Fixed peek-set-time-cmd  to take return as a valid :prompt-and-read.

;1; This is how long, in 60'ths of a second, to wait between updates of the screen.*
(DEFVAR 4*peek-sleep-time** 170) 

;1; Windows for PEEK.*

(DEFFLAVOR 4basic-peek* 
	   ((needs-redisplay nil))
	   (scroll-mouse-mixin
	    borders-mixin
	    scroll-bar-mixin
	    scroll-window-with-typeout full-screen-hack-mixin)
  :settable-instance-variables
  :gettable-instance-variables
  (:default-init-plist 
    :label "3Peek*"
    :truncation nil)
  (:documentation :special-purpose 
		  "3The actual peek window.  This has the capability to display in a PEEK display mode.*"))

(DEFMETHOD 4(basic-peek :who-line-documentation-string*) ()
  "2If over mouse sensitive item get its documentation, else print default string.*"
  (COND ((AND (CONSP tv:current-item)
              (CONSP (CDR tv:current-item)) 
              (GET (CDDR tv:current-item) :documentation)))
	(t '(:mouse-r-2 "3System Menu*"))))

(DEFMETHOD 4(basic-peek :name-for-selection*) ()
  (STRING-APPEND "3Peek: *" (label-string label)))

(DEFUN 4peek-mouse-click* (item leader-to-complement)
  (DECLARE (:self-flavor basic-peek))
  (SETQ needs-redisplay t)
  (SETF (ARRAY-LEADER item (+ scroll-item-leader-offset leader-to-complement))
        (NOT (ARRAY-LEADER item (+ scroll-item-leader-offset leader-to-complement)))))

(DEFFLAVOR 4dynamic-highlighting-command-menu-pane* ()
	   (w:menu)
  (:default-init-plist
    :highlighting nil
    :scrolling-p nil
    :command-menu t
    :dynamic t))


(DEFFLAVOR 4peek-frame*
	   ((peek-pane nil)	   ;1; These variables can be used in the commands to*
	    (typeout-pane nil)	   ;1; access the respective panes.*
	    (mode-pane nil)
	    (cmds-pane nil))
	   (ucl:command-loop-mixin
	    tv:stream-mixin
	    frame-dont-select-inferiors-with-mouse-mixin
	    bordered-constraint-frame-with-shared-io-buffer)
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables
  :special-instance-variables
  (:default-init-plist
    :active-command-tables '(peek-mode-cmd-table peek-cmd-table peek-other-cmd-table)
    :all-command-tables '(peek-mode-cmd-table peek-cmd-table peek-other-cmd-table)
    :menu-panes '((menu ucl-peek-menu)
		  (cmds ucl-peek-cmd-menu))
    :blip-alist '((:menu :handle-menu-input)
		  (:direct-command-entry :handle-direct-command-input)
		  (:mouse-button :peek-handle-blip)
	          (TELNET        :peek-handle-blip)
	          (QSEND         :peek-handle-blip)
	          (EH            :peek-handle-blip)
	          (INSPECT       :peek-handle-blip)
	          (DESCRIBE      :peek-handle-blip)
		  (host:host-status   :peek-handle-blip))       ;1; LS 11/06/86*
    :save-bits :delayed
    :typein-handler nil
    :basic-help '(peek-documentation-cmd)))

(DEFMETHOD 4(peek-frame :before :init*) (IGNORE)
  "2Set up the panes and constraints for the frame.*"
  (SETQ io-buffer (tv:make-default-io-buffer))
  (SETQ panes  `((PEEK basic-peek
		   :io-buffer ,io-buffer)
	         (menu dynamic-highlighting-command-menu-pane
		   :font-map ,(LIST fonts:cptfont)
                   :label (:string "3Modes*" :font fonts:hl10b)
		   :item-list nil
		   :io-buffer ,io-buffer)
                 (cmds dynamic-highlighting-command-menu-pane
		   :font-map ,(LIST fonts:cptfont)
                   :label (:string "3Commands*" :font fonts:hl10b)
		   :item-list nil
		   :io-buffer ,io-buffer)))
  (SETQ constraints `((main . ((PEEK menus)
			      ((menus  :horizontal (2 :lines menu) (menu cmds)
          		        ((cmds :ask :pane-size))
				((menu :even))))
			      ((PEEK :even)))))))

(DEFMETHOD 4(peek-frame :after :init*) (IGNORE)
  "2Set up the pane variables, which are used in the defcommands.*"
  (SETQ peek-pane (SEND self :get-pane 'PEEK)
	typeout-pane (SEND peek-pane :typeout-window)
	mode-pane (SEND self :get-pane 'menu)
	cmds-pane (SEND self :get-pane 'cmds))
  (SEND self :select-pane peek-pane)
  (colorize-peek-frame))

(DEFUN  4colorize-peek-frame* ()
  (DECLARE (:self-flavor peek-frame))
  (LET ((menu-pane (SEND self :get-pane 'menu))
	(cmd-pane (SEND self :get-pane 'cmds)))
    (SETF (sheet-background-color menu-pane) w:25%-gray-color)
    (SETF (label-background (SEND menu-pane :label)) w:50%-gray-color)
    (SETF (sheet-background-color cmd-pane) w:25%-gray-color)
    (SETF (label-background (SEND cmd-pane :label)) w:50%-gray-color)))
  

(DEFMETHOD 4(peek-frame :before :expose*) (&rest ignore)
  "2Necessary for the :ask :pane-size constraint for the menu.*"
  (OR exposed-p
      (SEND self :set-configuration 'main)))



(DEFMETHOD 4(peek-frame :designate-io-streams*) ()
  "2Redefine a UCL method to Bind io to the appropriate panes.*"
  (SETQ *terminal-io* typeout-pane
        *standard-output* typeout-pane
        *standard-input* typeout-pane
	*debug-io* typeout-pane))

(DEFMETHOD 4(peek-frame :around :loop*) (cont mt ignore)
  "2This is used to put up the first KEYBOARD status message, so the user knows were ready.*"
  (LET* ((kbd-intercepted-characters
	  (REMOVE (ASSOC #\Break kbd-intercepted-characters :test #'EQ)
		  kbd-intercepted-characters)))
    (UNLESS ucl::command-history
      (LET ((ch (tv:read-any *terminal-io*)))
	(SEND *terminal-io* :force-kbd-input ch)))
    (peek-clear-typeout peek-pane)
    (FUNCALL-WITH-MAPPING-TABLE cont mt :loop))) 

(DEFMETHOD 4(peek-frame :peek-handle-blip*) ()
  "2Handle special Peek blips.*"
  (LET ((blip ucl::kbd-input))
    (DECLARE (SPECIAL peek-pane))
    (CASE (CAR blip)
      (TELNET (TELNET (CADR blip)))
      (QSEND (QSEND (CADR blip))                      ;1; CHANGED 11/06/86 - LS - GOT RID OF EXTRA @*
             (SEND peek-pane :set-needs-redisplay t)
	     (SEND *standard-output* :make-complete))
      (EH (EH (CADR blip)))
      (INSPECT (INSPECT (CADR blip)))
      (DESCRIBE (DESCRIBE (CADR blip)))
      (host:host-status
       (IF (CADR blip)
	   (host:host-status (CADR blip))
	   (host:host-status)))    ;1; NEW - LS 11/06/86*
      (:mouse-button (BEEP))
      (otherwise
       (FERROR () "3Peek method :peek-handle-blip doesn't know what to do with blip ~A~%*" blip))))) 

(DEFMETHOD 4(peek-frame :before :fetch-input*) ()
  "2Before doing :any-tyi enter a while-no-typein - update-mode loop.*"
  (peek-when-typeout-query-then-clear peek-pane)
  (UNLESS ucl::command-execution-queue
    (LET* ((*terminal-io* (SEND peek-pane :typeout-window))
	   (sleep-time *peek-sleep-time*)
	   (wakeup-time (TIME-DIFFERENCE (TIME) (- sleep-time))))
      (LOOP (peek-timeout peek-pane wakeup-time sleep-time)
	 (WHEN (SEND *terminal-io* :listen)
	   (RETURN t))
	 (SEND peek-pane :redisplay)))))

(DEFMETHOD 4(peek-frame :around :handle-key-input*)
	   (cont mt ignore &optional ignore)
  "2Map uppercase characters to lowercase.*"
  (SETQ ucl:kbd-input (CHAR-DOWNCASE ucl:kbd-input))
  (FUNCALL-WITH-MAPPING-TABLE cont mt :handle-key-input ucl:kbd-input))

(DEFUN 4peek-timeout* (peek-pane wakeup-time sleep-time)
  "2Return when input or timeout.*"
  (WHEN (TIME-LESSP wakeup-time (TIME))
    (SETQ wakeup-time (TIME-DIFFERENCE (TIME) (- sleep-time))))
  (PROCESS-WAIT
    "3Peek timeout*"
    #'(lambda (wakeup-time stream frame)
	(AND (sheet-exposed-p frame)
	     (OR (TIME-LESSP wakeup-time (TIME))
		 (SEND stream :listen))))
    wakeup-time    
    (SEND peek-pane :typeout-window)
    (SEND peek-pane :superior)))

(DEFUN 4peek-wait-for-input* (typeout-pane)
  "2If there is typeout on top of the peek-pane this prompts the user for clearing it.*"
  (TERPRI typeout-pane)
  (FORMAT typeout-pane "3~& ~A*" *remove-typeout-standard-message*)
  (LET ((CHAR (tv:read-any *terminal-io*)))
    (UNLESS (EQL char #\Space)
      (tv:unread-any char *terminal-io*))))

(DEFUN 4peek-clear-typeout* (peek-pane)
  "2Remove the typeout pane from over the peek pane.*"
  (LET ((typeout-pane (SEND peek-pane :typeout-window)))
    (SEND typeout-pane :make-complete)
    (SEND peek-pane :redisplay)))

(DEFUN 4peek-when-typeout-query-then-clear* (peek-pane)
  "2When the typeout window is exposed query the user for clearing it.*"
  (LET ((typeout-pane (SEND peek-pane :typeout-window)))
    (WHEN (SEND typeout-pane :exposed-p)
      (peek-wait-for-input typeout-pane)
      (peek-clear-typeout peek-pane))))
  


;1; Commands for Peek modes.*

(DEFCOMMAND 4peek-processes-cmd* ()
  '(:description "3List status of every process -- why waiting, how much run recently.*"
    :names ("3Processes*")
    :keys (#\p))
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane))
  (LET ((menu-item (DOLIST (item ucl-peek-menu)
		     (WHEN (EQ self (THIRD item)) (RETURN item)))))
    (SEND mode-pane :set-highlighted-items (LIST menu-item))
    (SEND peek-pane :set-label "3Peek Processes*")
    (SEND peek-pane :set-display-item (peek-processes  nil))
   ))

(DEFCOMMAND 4peek-counters-cmd* ()
  '(:description "3Display the values of all the microcode meters.*"
    :names ("3Counters*")
    :keys (#\c #\%))
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane))
  (LET ((menu-item (DOLIST (item ucl-peek-menu)
		     (WHEN (EQ self (THIRD item)) (RETURN item)))))
    (SEND mode-pane :set-highlighted-items (LIST menu-item))
    (SEND peek-pane :set-label "3Statistic Counters*")
    (SEND peek-pane :set-display-item (peek-counters nil))))

(DEFCOMMAND 4peek-areas-cmd* ()
  '(:description "3Display status of areas, including how much memory allocated and used.*"
    :names ("3Areas*")
    :keys (#\a))
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane))
  (LET ((menu-item (DOLIST (item ucl-peek-menu)
		     (WHEN (EQ self (THIRD item)) (RETURN item)))))
    (SEND mode-pane :set-highlighted-items (LIST menu-item))
    (SEND peek-pane :set-label "3Areas*")
    (SEND peek-pane :set-display-item (peek-areas nil))))



(DEFCOMMAND 4peek-file-system-cmd* ()
  '(:description "3Display status of FILE protocol connections to remote file systems.*"
    :names ("3FILE Status*")
    :keys (#\f))
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane typeout-pane))
  (IF (FUNCTIONP 'fs::peek-file-system)
      (LET ((menu-item (DOLIST (item ucl-peek-menu)
		         (WHEN (EQ self (THIRD item)) (RETURN item)))))
        (SEND mode-pane :set-highlighted-items (LIST menu-item))
        (SEND peek-pane :set-label "3File System Status*")
        (SEND peek-pane :set-display-item (fs::peek-file-system nil)))
      (FORMAT typeout-pane "3There is no file system on this computer.~%*")))

(DEFCOMMAND 4peek-window-hierarchy-cmd* ()
  '(:description "3Display the hierarchy of window inferiors, saying which are exposed.*"
    :names ("3Windows*")
    :keys (#\w))
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane))
  (LET ((menu-item (DOLIST (item ucl-peek-menu)
		     (WHEN (EQ self (THIRD item)) (RETURN item)))))
    (SEND mode-pane :set-highlighted-items (LIST menu-item))
    (SEND peek-pane :set-label "3Window Hierarchy*")
    (SEND peek-pane :set-display-item (peek-window-hierarchy nil))))

(DEFCOMMAND 4peek-servers-cmd* ()
  '(:description "3List all servers, who they are serving, and their status.*"
    :names ("3Servers*")
    :keys (#\s))
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane))
  (LET ((menu-item (DOLIST (item ucl-peek-menu)
		     (WHEN (EQ self (THIRD item)) (RETURN item)))))
    (SEND mode-pane :set-highlighted-items (LIST menu-item))
    (SEND peek-pane :set-label "3Active Servers*")
    (SEND peek-pane :set-display-item (net::peek-servers nil))))



;1; Commands for Peek command menu.*

;1; Modified to fit into 604x432 window on MicroExplorer or 688x432 window on Explorer*
;1; Shortened command name so command window would size correctly - 11/19/87 CAT*
(DEFCOMMAND 4peek-histo-cmd* ()
  '(:description "3Display a statistical sampling of what functions use the most system resources.*"
    :names ("3Histogram*")
    :keys (#\m)) ;1; M for Meter, because F and H are already taken...*
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane))
    (IF (FBOUNDP 'meter::peek-histo)
      (LET ((menu-item (DOLIST (item ucl-peek-menu)
                         (WHEN (EQ self (THIRD item)) (RETURN item)))))
        (SEND mode-pane :set-highlighted-items (LIST menu-item))
        (SEND peek-pane :set-label "3Peek Functions*")
        (SEND peek-pane :set-display-item (meter::peek-histo nil))
        )
      (BEEP)))

(DEFCOMMAND 4peek-host-status-cmd* ()        ;1; LS  11/06/86*
  '(:description "3Print Network statistics of all known hosts.*"
    :names ("3Host Status*")
    :keys (#\h))
  (DECLARE (SPECIAL typeout-pane peek-pane))
  (LET ((*standard-output* typeout-pane))
    (IF (FUNCTIONP 'host:host-status)
	(host:host-status)
	(FORMAT t "3This computer is not on a network.~&*"))))
    
(DEFCOMMAND 4peek-set-time-cmd* ()
  '(:description "3Reset time between updates of the screen in units of 1/60ths of a second.*"
    :names ("3Set Timeout*")
    :keys (#\t))
  (DECLARE (SPECIAL typeout-pane peek-pane))
  (LET* ((*terminal-io* typeout-pane)
	 (num (UNWIND-PROTECT
		  (PROMPT-AND-READ
		    '(:number :or-nil t)  ;1; DAB 03-14-89 Take return, use default.*
		    "3~%Current timeout in 1/60ths of a second = ~A~%~%Enter new time: *" *peek-sleep-time*)
		(peek-clear-typeout peek-pane))))
    (WHEN num (SETQ *peek-sleep-time* num))))  ;1; DAB 03-14-89*

(DEFCOMMAND 4peek-exit-cmd* ()
  '(:description "3Exit the Peek process.*"
    :names ("3Exit*")
    :keys (#\end #\q))
  (DECLARE (SPECIAL peek-pane))
  (deselect-and-maybe-bury-window
    (SEND peek-pane :superior) :first))

(DEFCOMMAND 4peek-page-up-cmd* ()
  '(:description "3Display the next page up in the scroll window.*"
    :names ("3Page up*")
    :keys (#\m-v #\c-up-arrow #\rubout))
  (DECLARE (SPECIAL peek-pane))
  (SEND peek-pane :scroll-relative :top :bottom))

(DEFCOMMAND 4peek-page-down-cmd* ()
  '(:description "3Display the next page down in the scroll window.*"
    :names ("3Page down*")
    :keys (#\c-v #\c-down-arrow #\space))
  (DECLARE (SPECIAL peek-pane))
  (SEND peek-pane :scroll-relative :bottom :top))

(DEFCOMMAND 4peek-network-cmd* ()             ;1; CHANGED NAME -LS  - 11/06/86*
  '(:description "3Display useful information about all network connections & other network info.*"
    :names ("3Network*")
    :keys (#\n))
  (DECLARE (SPECIAL ucl-peek-menu peek-pane mode-pane typeout-pane))
  (IF (FUNCTIONP 'net::peek-network)         ;1; USED TO BE CHAOS  -- LS*
      (LET ((menu-item (DOLIST (item ucl-peek-menu)
		         (WHEN (EQ self (THIRD item)) (RETURN item)))))
        (SEND mode-pane :set-highlighted-items (LIST menu-item))
        (SEND peek-pane :set-label "3Network*")
        (SEND peek-pane :set-display-item (net::peek-network nil)))    ;1; PUT IN NET PACKAGE  -- LS*
      (FORMAT t "3This computer is not on a network.~%*")))


;1; Modified to fit into 604x432 window on MicroExplorer or 688x432 window on Explorer*
;1; Shortened a line of text and  changed command name from documentation to help  - 11/19/87 CAT*
(DEFCOMMAND 4peek-documentation-cmd* ()
  '(:description "3Print out some brief general documentation about Peek.*"
    :names ("3Help*")
    :keys (#\C-help #\M-help))
  (DECLARE (SPECIAL typeout-pane peek-pane))
  (LET ((*standard-output* typeout-pane))
    (FORMAT t
 "
                                PEEK HELP


PEEK is a utility composed of MODES and COMMANDS.

MODES contain status information about various components of the Explorer 
system.  They are listed at the bottom of the screen and are briefly described 
below.  A MODE can be displayed by highlighting it with the mouse cursor and 
then clicking Mouse-Left or by entering its key assignment.

     *  WINDOWS--Displays the hierarchy of window inferiors, depicting which
                 are exposed.  Key assignment: w

     *  AREAS--Displays status of areas, including how much memory is allocated
               and used.  Key assignment: a

     *  PROCESSES--Lists status of every process.  Key assignment:  p

     *  FILE STATUS--Displays status of file protocol connections to remote file
                     systems.  Key assignment: f

     *  SERVERS--Lists all servers, who they are serving, and their status.
                 Key assignment: s

     *  NETWORK--Displays network connection information.  Key assignment: n

     *  COUNTERS--Displays the value of all microcode meters.  Key assignment: c

     *  FUNCTION HISTOGRAM--Displays most frequently called functions in one or
                           more processes. Key assignment: m
 
COMMANDS are actions that can be performed.  They are listed at the bottom
of the screen in the right-hand corner and are briefly described below.

     *  DOCUMENTATION--Displays this message.  Key assignment: CTRL-HELP

     *  SET TIMEOUT--Reset time between updates of the screen in units of 1/60ths  
                     of a second.  Key assignment: t

     *  HOSTAT--Displays network statistics of all known hosts.  Key assignment: h

     *  EXIT--Exits the Peek utility.  Key assignment:  or q


 ")))

(DEFCOMMAND 4peek-break-cmd* ()
  '(:description "3Enter the break read, eval and print loop.*"
    :names ("3Break*")
    :keys (#\Break))
  (DECLARE (SPECIAL typeout-pane peek-pane))
  (LET ((*terminal-io* typeout-pane))
    (BREAK "3Peek*" t)
    (peek-clear-typeout peek-pane)))



(DEFPARAMETER 4peek-modes*
   '(peek-window-hierarchy-cmd
      peek-processes-cmd
      peek-servers-cmd
      peek-counters-cmd
      peek-areas-cmd
      peek-file-system-cmd
      peek-network-cmd
      peek-histo-cmd)) 

(DEFPARAMETER 4peek-cmds*
   '(peek-documentation-cmd
      peek-host-status-cmd   ;1; LS*
      peek-set-time-cmd
      peek-exit-cmd)) 

(DEFPARAMETER 4peek-non-menu-cmds*
              '(peek-page-up-cmd
                 peek-page-down-cmd
                 peek-break-cmd)) 

(BUILD-COMMAND-TABLE 'peek-mode-cmd-table 'peek-frame
  peek-modes
  :init-options
  '(:name "3Peek modes*"))

(BUILD-COMMAND-TABLE 'peek-cmd-table 'peek-frame
  peek-cmds
  :init-options
  '(:name "3Peek commands*"))

(BUILD-COMMAND-TABLE 'peek-other-cmd-table 'peek-frame
  peek-non-menu-cmds
  :init-options
  '(:name "3Other Peek commands*"))

(BUILD-MENU 'ucl-peek-menu 'peek-frame
  :default-item-options '(:font fonts:cptfont)
  :item-list-order peek-modes) 

(BUILD-MENU 'ucl-peek-cmd-menu 'peek-frame
  :default-item-options '(:font fonts:cptfont)
  :item-list-order peek-cmds) 

(COMPILE-FLAVOR-METHODS peek-frame basic-peek dynamic-highlighting-command-menu-pane) 


(DEFUN 4peek* (&optional initial-mode)
  "2Select a new or old Peek window.  An argument sets the Peek display mode.*"
  (select-or-create-window-of-flavor 'peek-frame)
  (IF initial-mode
    (SEND selected-window :force-kbd-input
       (TYPECASE initial-mode
	 (STRING (AREF initial-mode 0))
	 (:symbol (AREF (SYMBOL-NAME initial-mode) 0))
	 (t initial-mode))))
  (await-window-exposure)) 

;1; Modified to fit into 604x432 window on MicroExplorer or 688x432 window on Explorer*
;1; Shortened name from 30 to 28, state from 21 to 19 chars, % from 8 to 7 - 11/24/87 CAT*
(DEFUN 4peek-processes* (peek-pane &rest ignore)
  "2Shows state of all active processes.*"
  (LIST ()
	;1; 30 of process name, 25 of state, 5 of priority, 10 of quantum left/quantum,*
	;1; 8 of percentage, followed by idle time (11 columns)*
	(scroll-parse-item
	 (FORMAT () "3~28A~19A~10A~10A~7A~8A*" "3Process Name*" "3State*" "3Priority*" "3Quantum*" "3 %*"
		 "3Idle*"))
	(scroll-parse-item "")
	(scroll-maintain-list #'(lambda ()
				  all-processes)
			      #'(lambda (process)
				  (scroll-parse-item
				   `(:mouse-item
				     (nil :eval (peek-process-menu ',process ',peek-pane 'item 0)
				      :documentation
				      "3Menu of useful things to do to this process.*")
				     :string ,(PROCESS-NAME process) 34)
				   `(:function ,(FUNCTION peek-whostate) ,(CONS process ()) 23)
				   `(:function ,process (:priority) 11 ("3~6D.*"))
				   `(:function ,process (:quantum-remaining) 5 ("3~4D/*"))
				   `(:function ,process (:quantum) 5 ("3~D.*"))
				   `(:function ,process (:percent-utilization) 7
                                               ("3~1,1,4$*"))
				   `(:function ,process (:idle-time) nil
                                               ("3~\\PEEK-PROCESS-IDLE-TIME\\*"))))
			      () ())
	(scroll-parse-item "")
        (scroll-parse-item "3Clock Function List*")
	(scroll-maintain-list #'(lambda () clock-function-list)
			      #'(lambda (func)
				  (scroll-parse-item
				   `(:string ,(WITH-OUTPUT-TO-STRING (str)
						(PRINC func str)))))))
	  ) 

(DEFPROP 4format::peek-process-idle-time* peek-process-idle-time format::format-ctl-one-arg) 

(DEFUN 4peek-process-idle-time* (arg ignore)
  (COND ((NULL arg) (SEND *standard-output* :string-out "3forever*"))	;1 character too small*
	((ZEROP arg))				;1; Not idle*
	((< arg 60.) (FORMAT *standard-output* "3~2D sec*" arg))
	((< arg 3600.) (FORMAT *standard-output* "3~2D min*" (TRUNCATE arg 60.)))
	(t (FORMAT *standard-output* "3~2D hr*" (TRUNCATE arg 3600.)))))

(DEFUN 4peek-whostate* (process)
  (COND ((si::process-arrest-reasons process) "3Arrest*")
	((si::process-run-reasons process) (PROCESS-WHOSTATE process))
	(t "3Stop*")))


(DEFUN 4peek-counters* (IGNORE)
  "2Statistics counters*"
  (LIST ()
    (scroll-maintain-list #'(lambda () sys:a-memory-counter-block-names)
			  #'(lambda (counter)
			      (scroll-parse-item
				`(:string ,(STRING counter) 35.)
				`(:function read-meter (,counter) nil ("3~@15A*" 10. t)))))))

;1;; Memory*
;1;; Modified to fit into 604x432 window on MicroExplorer or 688x432 window on Explorer*
;1;; changed from one line entry to three lines - 11/19/87 CAT*
(DEFUN 4peek-memory-header* ()
  (LIST ()
  (scroll-parse-item
      "3Physical memory: *"
      `(:function ,#'(lambda (&aux (val (system-communication-area %sys-com-memory-size)))
		       (SETF (value 0) (TRUNCATE val 2000))
		       val)
		  nil nil (nil 8.))
      `(:value 0 nil ("3 (~DK)*")))
  (scroll-parse-item
      "3Free space:      *"
      `(:function ,#'(lambda (&aux (val (sys::usable-address-space)))
		       (SETF (value 0) (TRUNCATE val 2000))
		       val)
		  nil nil (nil 8.))
      `(:value 0 nil ("3 (~DK)*")))
  (scroll-parse-item
      "3Wired pages:     *"
      `(:function ,#'(lambda ()
		       (MULTIPLE-VALUE-BIND (n-wired-pages n-fixed-wired-pages)
			   (si::count-wired-pages)
			 (SETF (value 0) (- n-wired-pages n-fixed-wired-pages))
			 (SETF (value 1) (TRUNCATE n-wired-pages (TRUNCATE 2000 page-size)))
			 (SETF (value 2) (REM n-wired-pages (TRUNCATE 2000 page-size)))
			 n-fixed-wired-pages))
		  nil nil ("3~D*"))
      `(:value 0 nil ("3+~D *"))
      `(:value 1 nil ("3(~D*"))
      `(:value 2 nil ("3~[~;.25~;.5~;.75~]K)*")))))


;1;; Modified to fit into 604x432 window on MicroExplorer or 688x432 window on Explorer*
;1;; Shortened area name from 40 to 33 chars, region from 15 to 13 chars - 11/19/87 CAT*
(DEFUN 4peek-areas* (IGNORE)
  "2Areas*"
  (LIST () (peek-memory-header) (scroll-parse-item "")
	(scroll-maintain-list #'(lambda ()
				  0)
			      #'(lambda (area)
				  (LIST '(:pre-process-function peek-areas-region-display)
					(scroll-parse-item :mouse-self
							   '(nil :eval
							     (peek-mouse-click 'self 0)
							     :documentation
							     "3Insert//remove display of all regions in this area.*")
							   :leader `(nil ,area)
							   `(:function identity (,area) 4
							     ("3~3D*"))
							   `(:string ,(STRING (AREA-NAME area))
							     41)
							   `(:function
							     ,(FUNCTION
							       (lambda (area)
								 (MULTIPLE-VALUE-BIND (LENGTH used n-regions)
								   (si::room-get-area-length-used
								    area)
								   (SETF (value 0) used)
								   (SETF (value 1) length)
								   (SETF (value 2)
									 (COND
									   ((ZEROP length) 0)
									   ((< length 40000)
									    (TRUNCATE
									     (* 144
										(- length used))
									     length))
									   (t
									    (TRUNCATE
									     (- length used)
									     (TRUNCATE length
										       144)))))
								   n-regions)))
							     (,area) 15 ("3(~D region~:P)*"))
							   '(:value 2 nil ("3~@3A% free, *" 12 t))
							   '(:value 0 nil ("3~O*"))
							   '(:value 1 nil ("3/~O used*")))))
			      ()
			      #'(lambda (state)
				  (PROG (next-one
					 this-one
					 (len (ARRAY-TOTAL-SIZE #'AREA-NAME)))
				    (DO ((i state (1+ i)))
					((>= i len)
					 nil)
				      (COND
					((AND (NULL this-one) (AREF #'AREA-NAME i))
					 (SETQ this-one i))
					((AND this-one (AREF #'AREA-NAME i)) (SETQ next-one i)
					 (RETURN t))))
				    (RETURN (VALUES this-one next-one (NULL next-one)))))))) 

(DEFUN 4peek-areas-region-display* (item)
  "2Handles adding/deleting of the region display when a mouse button is clicked.*"
  (COND
    ((NULL (ARRAY-LEADER (CADR item) scroll-item-leader-offset)))
    ;1; Clicked on this item, need to complement state*
    ((= (LENGTH item) 2)
     ;1; If aren't displaying regions now, display them*
     (RPLACD (CDR item)
	     (CONS
	      (scroll-maintain-list
	       #'(lambda ()
		   (area-region-list
		    (ARRAY-LEADER (FIRST (tv:scroll-item-component-items item)) (1+ scroll-item-leader-offset))))
	       #'(lambda (region)
		   (scroll-parse-item
		    `(:string
		      ,(FORMAT () "3~d:~6TOrigin #o~O,~25TLength ~D, *" region    ;1 DAB 01-19-89*
			       (si::region-origin-true-value region) (region-length region))
		      47)
		    `(:function ,(FUNCTION region-free-pointer) (,region) 16 ("3Used ~D, *"))
		    `(:function ,(FUNCTION region-gc-pointer) (,region) 14 ("3GC #o~O, *")) ;1 DAB 01-19-89*
		    `(:function
		      ,(FUNCTION
			(lambda (region &aux bits)
			  (SETQ bits (region-bits region))
			  (SETF (value 0)
				(NTH (LDB %%region-space-type bits)
				     '("3FREE*" "3OLD*" "3NEW*" "3NEW1*" "3NEW2*" "3NEW3*" "3NEW4*" "3NEW5*" "3NEW6*"
		                     "3STATIC*" "3FIXED*" "3EX-PDL*" "3COPY*" "3TRAIN*" "3TYPE16*" "3TYPE17*")))

			  (SETF (value 1) (LDB %%region-map-bits bits))
			  (SETF (value 2) (LDB %%region-scavenge-enable bits))
			  (SETF (value 3) (SELECT (LDB %%region-usage bits)
					    (%region-usage-active :active)
					    (%region-usage-inactive-1 :inactive-1)
					    (%region-usage-inactive-2 :inactive-2)
					    (%region-usage-inactive-3 :inactive-3)))
			  (SETF (value 4) (LDB %%region-generation bits))
			  (NTH (LDB %%region-representation-type bits)
			       '(LIST struc "3REP=2*" "3REP=3*"))))
		      (,region) 13 ("3Type ~A *"))
		    '(:value 0 12 ("3~A, *")) '(:value 1 nil ("3Map #o~O, *")) ;1 DAB 01-19-89*
				      '(:value 2 nil ("3~[NoScav~;Scav~]*"))
				      '(:value 3 nil ("3 ~o,*"))
				      '(:value 4 nil ("3 Gen ~d*")) ;1 DAB 01-19-89*
				      ))
	       ()
	       #'(lambda (state)
		   (BLOCK ()
		     (RETURN
		      (VALUES state (region-list-thread state)
			      (MINUSP (region-list-thread state)))))))
	      ())))
    (t (RPLACD (CDR item) ())))
  (SETF (ARRAY-LEADER (CADR item) scroll-item-leader-offset) ())) 


(DEFUN 4peek-process-menu* ( &rest args)
  (APPLY #'PROCESS-RUN-FUNCTION "3Peek Process Menu*"
         self :peek-process-menu args)) 

(DEFUN 4choose-new-priority* (process)
  (LET ((*print-base* 12)
	(*read-base* 12)
	(*nopoint t)
	(priority (SEND process :priority)))
    (DECLARE (SPECIAL priority))
       (choose-variable-values
	'((priority "3Priority*" :documentation "3Set priority of process*" :number))
        :label (FORMAT () "3Priority for ~A*" process)
        :margin-choices `(, w::margin-choice-completion-string
			  (, w::margin-choice-abort-string (SIGNAL-CONDITION eh:*abort-object*))))
			  priority))

(DEFUN 4choose-new-arrest-reason* (process)
  (LET ((*print-base* 12)
	(*read-base* 12)
	(*nopoint t)
	(arrest-reason :user))
    (DECLARE (SPECIAL arrest-reason))
       (choose-variable-values
	'((arrest-reason "3Arrest Reason*" :documentation "3Choose an arrest reason*" :sexp))
        :label (FORMAT () "3Arrest reason for ~A*" process)
        :margin-choices `(, w::margin-choice-completion-string
			  (, w::margin-choice-abort-string (SIGNAL-CONDITION eh:*abort-object*))))
      arrest-reason)) 

(DEFUN 4choose-arrest-reason* (process &aux chosen-reason)
  (LET ((arrest-reasons (si::process-arrest-reasons process)))
    (WHEN arrest-reasons
      (LET ((alist))
	(DOLIST (reason arrest-reasons)
	  (SETQ alist
		(CONS (CONS reason `(:value ,reason :documentation "3Remove this arrest reason from this process.*"))
		      alist)))
	(SETQ alist (CONS '(all :value :all :documentation "3Remove all arrest reasons from this process.*") alist))
	(SETQ chosen-reason (w:menu-choose alist :label (FORMAT nil "3Select reason to unarrest ~A*" process)))))
    (IF (EQ chosen-reason :all) arrest-reasons chosen-reason)))



(DEFMETHOD 4(basic-peek :peek-process-menu*) (process &rest ignore &aux choice)
  "2Menu for interesting operations on processes in a peek display*"
  (LET ((*terminal-io* typeout-window)
	(choices
	 '(("3Debugger*" :value process-eh
            :documentation
	    "3Call the debugger to examine the selected process.*")
	   ("3Arrest*" :value process-arrest
            :documentation
	    "3Arrest the selected process.  Undone by Un-Arrest.*")
	   ("3Un-Arrest*" :value process-un-arrest
            :documentation
	    "3Un-Arrest the selected process.  Complement of Arrest.*")
	   ("3Flush*" :value process-flush
            :documentation
	    "3Unwind the selected process' stack and make it unrunnable.  Ask for confirmation.*")
	   ("3Reset*" :value process-reset
            :documentation
	    "3Reset the selected process.  Ask for confirmation.*")
           ("3Reset & Enable*" :value process-reset-enable
            :documentation
            "3Reset and Enable the selected process. Ask for confirmation.*")
	   ("3Kill*" :value process-kill
            :documentation
	    "3Kill the selected process.  Ask for confirmation.*")
	   ("3Describe*" :value process-describe
            :documentation
            "3Call DESCRIBE on this process.*")
	   ("3Priority*" :value process-priority
            :documentation
	    "3Change priority of the selected process.*")
	   ("3Inspect*" :value process-inspect
            :documentation
            "3Call INSPECT on this process.*"))))
 	;1; Don't offer EH for a simple process.*
    (OR (TYPEP (PROCESS-STACK-GROUP process) 'STACK-GROUP) (POP choices))
    (SETQ choice (w:menu-choose choices :label (PROCESS-NAME process) :scrolling-p nil))
    (CASE choice
      (process-arrest
       (LET ((new-reason (choose-new-arrest-reason process)))
	 (IF new-reason
	   (SEND process :arrest-reason new-reason))))
      (process-un-arrest
       (LET ((arrest-reasons (choose-arrest-reason process)))
	 (COND
	   ((NULL arrest-reasons) nil)
	   ((LISTP arrest-reasons)
	    (LOOP for reason in arrest-reasons do (SEND process :revoke-arrest-reason reason)))
	   (t (SEND process :revoke-arrest-reason arrest-reasons)
	   )
	   )))
      (process-flush (IF (mouse-y-or-n-p (FORMAT () "3Flush ~A*" process))
		       (SEND process :flush))
		     )
      (PROCESS-RESET (IF (mouse-y-or-n-p (FORMAT () "3Reset ~A*" process))
		       (SEND process :reset))
		     )
      (process-reset-enable (IF (mouse-y-or-n-p (FORMAT () "3Reset & Enable ~A*" process))
                                (PROCESS-RESET-AND-ENABLE process))
			    )
      (process-kill (IF (mouse-y-or-n-p (FORMAT () "3Kill ~A*" process))
		      (SEND process :kill))
		    )
      (process-eh (SEND self :force-kbd-input `(EH ,process)))
      (process-describe (SEND self :force-kbd-input `(DESCRIBE ,process)))
      (process-priority
       (LET ((new-priority (choose-new-priority process)))
	 (IF new-priority
	   (SEND process :set-priority new-priority))))
      (process-inspect (SEND self :force-kbd-input `(INSPECT ,process)))
      (nil)
      (otherwise (BEEP)))))




(DEFUN 4peek-window-hierarchy* (IGNORE)
  (scroll-maintain-list #'(lambda () all-the-screens)
			#'(lambda (screen)
			    (LIST ()
			      (scroll-parse-item (FORMAT nil "3Screen ~A*" screen))
			      (peek-window-inferiors screen 2)
			      (scroll-parse-item "")))))

(DEFUN 4peek-window-inferiors* (window indent)
  (DECLARE (SPECIAL window indent))
  (scroll-maintain-list (CLOSURE '(window) #'(lambda () (sheet-inferiors window)))
			(CLOSURE '(indent)
			  #'(lambda (sheet)
			      (LIST ()
				    (scroll-parse-item 
				      (FORMAT nil "3~V@t*" indent)
				      `(:mouse
					 (nil :eval (peek-window-menu ',sheet)
					      :documentation
					      "3Menu of useful things to do to this window.*")
					 :string
					 ,(STRING (SEND sheet :name))))
				    (peek-window-inferiors sheet (+ indent 4)))))))

(DEFUN 4peek-window-menu* (&rest args)
  (APPLY #'PROCESS-RUN-FUNCTION "3Peek Window Menu*"
         self :peek-window-menu args)) 

(DEFMETHOD 4(basic-peek :peek-window-menu*) (sheet &rest ignore &aux choice)
  "2Menu for interesting operations on sheets in a peek display*"
  (SETQ choice
	(w:menu-choose
	 '(("3Deexpose*" :value :deexpose :documentation "3Deexpose the window.*")
	   ("3Expose*" :value :expose :documentation "3Expose the window.*")
	   ("3Select*" :value :select :documentation "3Select the window.*")
	   ("3Deselect*" :value :deselect :documentation "3Deselect the window.*")
	   ("3Deactivate*" :value :deactivate :documentation "3Deactivate the window.*")
	   ("3Kill*" :value :kill :documentation "3Kill the window.*")
	   ("3Bury*" :value :bury :documentation "3Bury the window.*")
	   ("3Describe*" :value describe :documentation "3Describe the window.*")
	   ("3Inspect*" :value inspect :documentation "3Inspect the window.*"))
	 :label (SEND sheet :name) :scrolling-p nil))
  (AND choice (OR (NEQ choice :kill) (mouse-y-or-n-p (FORMAT () "3Kill ~A*" (SEND sheet :name))))
     (COND
       ((MEMBER choice '(INSPECT describe) :test #'EQ)
	(LET ((*terminal-io* typeout-window))
	  (SEND self :force-kbd-input `(,choice ,sheet))))
       (t (SEND sheet choice)
	  )))) 

;1(tv:remove-system-key #\p)*			1; In case one already exits.*
;1(tv:add-system-key #\p 'tv:peek-frame "Peek - display system activities and their current states." t)*

;1(tv:delete-from-system-menu-column :debug "Peek")*	1; In case one already exists.*
;1(tv:add-to-system-menu-column :debug "Peek" '(SELECT-OR-CREATE-WINDOW-OF-FLAVOR 'TV:PEEK-FRAME)*
;			1      "Display system activities and their current states.")
